<?php
/**
 * PrestaPicqer - PrestaShop Module
 * 
 * @author     SofTer B.V.
 * @copyright  2025 SofTer B.V.
 * @license    Commercial License - https://softerbv.nl/prestapicqer/docs/license.txt
 * @version    1.0.8
 * 
 * This file is part of PrestaPicqer, a proprietary PrestaShop module.
 * Unauthorized copying, distribution, or modification is strictly prohibited.
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Class PrestapicqerWebhookModuleFrontController
 * Handles incoming webhook requests for stock and order status updates.
 */
class PrestapicqerWebhookModuleFrontController extends ModuleFrontController
{
    
    /**
     * This method is called for POST requests to the front controller and handels synchronise_stock requests.
     */
    public function postProcess()
    {
        // Set content type to JSON for the response
        header('Content-Type: application/json');

        if (!isset($_SERVER['REQUEST_METHOD']) || Tools::strtoupper($_SERVER['REQUEST_METHOD']) !== 'POST') {
            if(isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'GET' && !empty(Tools::getValue('action')) && Tools::getValue('action') === 'synchronise_stock'){
                if($this->syncFullCatalogStock()){
                    $this->sendJsonResponse(true, 'Done synchronising stock.', 200);
                }
                else{
                    $this->sendJsonResponse(false, 'Failed to synchronise stock.', 500);
                }
            } 
            else {
                $this->sendJsonResponse(false, 'Only POST requests are allowed.', 405);
            }
            return;
        }

        // Get raw POST data (JSON payload) before decoding
        $webhookPayloadRaw = file_get_contents('php://input');

        // --- Picqer Webhook Signature Validation ---
        $picqerSignature = isset($_SERVER['HTTP_X_PICQER_SIGNATURE']) ? $_SERVER['HTTP_X_PICQER_SIGNATURE'] : '';
        $configuredSecret = Configuration::get(PrestaPicqer::PRESTAPICQER_WEBHOOK_SECRET);

        if (empty($configuredSecret)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Webhook Secret is not configured. Cannot validate signature.',
                3, // Error
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Webhook secret not configured.', 500);
            return;
        }

        if (empty($picqerSignature)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing X-Picqer-Signature header. Request rejected.',
                3, // Error
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Unauthorized: Missing signature.', 401);
            return;
        }

        $checkingSignature = base64_encode(
            hash_hmac('sha256', $webhookPayloadRaw, $configuredSecret, true)
        );

        if (!hash_equals($checkingSignature, $picqerSignature)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Invalid X-Picqer-Signature. Request rejected. Received: ' . $picqerSignature . ', Expected: ' . $checkingSignature,
                3, // Error
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Unauthorized: Invalid signature.', 401);
            return;
        }
        // --- End Picqer Webhook Signature Validation ---

        // Now that signature is validated, decode the JSON payload
        $data = json_decode($webhookPayloadRaw, true);


        // Check for JSON parsing errors after validation
        if (json_last_error() !== JSON_ERROR_NONE) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Invalid JSON payload received after signature validation. Raw data: ' . $webhookPayloadRaw,
                3, // Error
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Invalid JSON payload.', 400);
            return;
        }

        // --- Handle different webhook events ---
        if (!isset($data['event'])) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing "event" field in payload. Payload: ' . json_encode($data),
                2, // Warning
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Missing event type.', 400);
            return;
        }

        switch ($data['event']) {
            case 'products.free_stock_changed':
                $this->processStockUpdateWebhook($data);
                break;
            case 'products.assembled_stock_changed':
                $this->processAssembledStockUpdateWebhook($data);
                break;
            case 'orders.status_changed':
                $this->processOrderStatusWebhook($data);
                break;
            default:
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Unhandled event type: ' . $data['event'] . '. Payload: ' . json_encode($data),
                    1, // Info
                    null,
                    'PrestaPicqer'
                );
                $this->sendJsonResponse(true, 'Event type ' . $data['event'] . ' received but not handled.', 200);
                break;
        }

        
    }

    /**
     * Processes a product stock update webhook from Picqer.
     */
    protected function processStockUpdateWebhook($data)
    {
        $pull_stock_enabled = (bool) Configuration::get(PrestaPicqer::PRESTAPICQER_PULL_STOCK_ENABLED);
        $pull_stock_full_sync = (bool) Configuration::get(PrestaPicqer::PRESTAPICQER_PULL_STOCK_FULL_SYNC);

        if (!$pull_stock_enabled) {
            // PrestaShopLogger::addLog(
            //     'PrestaPicqer Webhook: Pull Product Stock is disabled. Skipping stock update.',
            //     1, null, 'PrestaPicqer'
            // );
            $this->sendJsonResponse(true, 'Pull Product Stock is disabled.', 200);
            return;
        }

        if ($pull_stock_full_sync) {
            // PrestaShopLogger::addLog(
            //     'PrestaPicqer Webhook: Full catalog sync triggered by products.free_stock_changed webhook. This might be slow.',
            //     2, null, 'PrestaPicqer'
            // );
            $this->syncFullCatalogStock();
            $this->sendJsonResponse(true, 'Full catalog stock sync initiated.', 200);
            return;
        }

        // Process single product stock update
        if (!isset($data['data']) || !isset($data['data']['stock']) || !is_array($data['data']['stock'])) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing "data" key or "stock" array within "data" in products.free_stock_changed payload. Payload: ' . json_encode($data),
                3, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Missing product data or stock array.', 400);
            return;
        }

        $product_reference = isset($data['data']['productcode']) ? pSQL($data['data']['productcode']) : '';
        $total_freestock = 0;
        foreach ($data['data']['stock'] as $stock_entry) {
            if (isset($stock_entry['freestock'])) {
                $total_freestock += (int) $stock_entry['freestock'];
            }
        }
        $new_quantity = $total_freestock < 0 ? 0 : $total_freestock;


        $id_product_attribute = 0; // Default

        if (empty($product_reference)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing or invalid product_reference in stock update payload. Payload: ' . json_encode($data),
                3, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(true, 'Missing or invalid product_reference.', 200);
            return;
        }

        $product_info = $this->getProductIdAndAttributeIdByReference($product_reference);

        if (!$product_info || empty($product_info['id_product'])) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Product not found by reference "' . $product_reference . '". Payload: ' . json_encode($data),
                2, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(true, 'Product not found with provided reference.', 200);
            return;
        }

        $id_product = (int) $product_info['id_product'];
        $id_product_attribute = (int) $product_info['id_product_attribute'];

        $id_shop = Context::getContext()->shop->id;

        try {
            $old_quantity = Product::getQuantity($id_product, $id_product_attribute);

            $set_quantity_result = StockAvailable::setQuantity(
                $id_product,
                $id_product_attribute,
                $new_quantity,
                $id_shop
            );

            $updated_quantity = Product::getQuantity($id_product, $id_product_attribute);
            $is_successful = ($updated_quantity == $new_quantity);

            if ($is_successful) {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Stock updated for Product ID ' . $id_product .
                    ($id_product_attribute ? ' (Attribute ID ' . $id_product_attribute . ')' : '') .
                    ' from ' . $old_quantity . ' to ' . $updated_quantity . ' units (Target: ' . $new_quantity . ').',
                    1, null, 'PrestaPicqer', $id_product
                );
                $this->sendJsonResponse(true, 'Stock updated successfully for Product ID ' . $id_product . '.', 200);
            } else {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Stock update for Product ID ' . $id_product .
                    ' (Attribute ID ' . $id_product_attribute . ') failed to reach target quantity. ' .
                    'Target: ' . $new_quantity . ', Actual: ' . $updated_quantity . '. ' .
                    'StockAvailable::setQuantity returned: ' . ($set_quantity_result ? 'true' : 'false') . '. Payload: ' . json_encode($data),
                    3, null, 'PrestaPicqer', $id_product
                );
                $this->sendJsonResponse(false, 'Failed to update stock: Quantity mismatch after operation.', 500);
            }
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Exception during stock update for Product ID ' . $id_product .
                '. Error: ' . $e->getMessage() . '. Payload: ' . json_encode($data),
                3, null, 'PrestaPicqer', $id_product
            );
            $this->sendJsonResponse(false, 'An error occurred during stock update: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Processes a product stock update webhook from Picqer.
     */
    protected function processAssembledStockUpdateWebhook($data)
    {
        $pull_stock_enabled = (bool) Configuration::get(PrestaPicqer::PRESTAPICQER_PULL_STOCK_ENABLED);
        $pull_stock_full_sync = (bool) Configuration::get(PrestaPicqer::PRESTAPICQER_PULL_STOCK_FULL_SYNC);

        if (!$pull_stock_enabled) {
            // PrestaShopLogger::addLog(
            //     'PrestaPicqer Webhook: Pull Product Stock is disabled. Skipping stock update.',
            //     1, null, 'PrestaPicqer'
            // );
            $this->sendJsonResponse(true, 'Pull Product Stock is disabled.', 200);
            return;
        }

        if ($pull_stock_full_sync) {
            // PrestaShopLogger::addLog(
            //     'PrestaPicqer Webhook: Full catalog sync triggered by products.free_stock_changed webhook. This might be slow.',
            //     2, null, 'PrestaPicqer'
            // );
            $this->syncFullCatalogStock();
            $this->sendJsonResponse(true, 'Full catalog stock sync initiated.', 200);
            return;
        }

        // Process single product stock update
        if (!isset($data['data']) || !isset($data['data']['stock']) || !is_array($data['data']['stock'])) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing "data" key or "stock" array within "data" in products.free_stock_changed payload. Payload: ' . json_encode($data),
                3, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Missing product data or stock array.', 400);
            return;
        }

        $product_reference = isset($data['data']['productcode']) ? pSQL($data['data']['productcode']) : '';
        $total_freestock = 0;
        foreach ($data['data']['stock'] as $stock_entry) {
            if (isset($stock_entry['freestock'])) {
                $total_freestock += (int) $stock_entry['freestock'];
            }
        }
        $new_quantity = $total_freestock < 0 ? 0 : $total_freestock;


        $id_product_attribute = 0; // Default

        if (empty($product_reference)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing or invalid product_reference in stock update payload. Payload: ' . json_encode($data),
                3, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(true, 'Missing or invalid product_reference.', 200);
            return;
        }

        $product_info = $this->getProductIdAndAttributeIdByReference($product_reference);

        if (!$product_info || empty($product_info['id_product'])) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Product not found by reference "' . $product_reference . '". Payload: ' . json_encode($data),
                2, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(true, 'Product not found with provided reference.', 200);
            return;
        }

        $id_product = (int) $product_info['id_product'];
        $id_product_attribute = (int) $product_info['id_product_attribute'];

        $id_shop = Context::getContext()->shop->id;

        try {
            $old_quantity = Product::getQuantity($id_product, $id_product_attribute);

            $set_quantity_result = StockAvailable::setQuantity(
                $id_product,
                $id_product_attribute,
                $new_quantity,
                $id_shop
            );

            $updated_quantity = Product::getQuantity($id_product, $id_product_attribute);
            $is_successful = ($updated_quantity == $new_quantity);

            if ($is_successful) {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Stock updated for Product ID ' . $id_product .
                    ($id_product_attribute ? ' (Attribute ID ' . $id_product_attribute . ')' : '') .
                    ' from ' . $old_quantity . ' to ' . $updated_quantity . ' units (Target: ' . $new_quantity . ').',
                    1, null, 'PrestaPicqer', $id_product
                );
                $this->sendJsonResponse(true, 'Stock updated successfully for Product ID ' . $id_product . '.', 200);
            } else {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Stock update for Product ID ' . $id_product .
                    ' (Attribute ID ' . $id_product_attribute . ') failed to reach target quantity. ' .
                    'Target: ' . $new_quantity . ', Actual: ' . $updated_quantity . '. ' .
                    'StockAvailable::setQuantity returned: ' . ($set_quantity_result ? 'true' : 'false') . '. Payload: ' . json_encode($data),
                    3, null, 'PrestaPicqer', $id_product
                );
                $this->sendJsonResponse(false, 'Failed to update stock: Quantity mismatch after operation.', 500);
            }
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Exception during stock update for Product ID ' . $id_product .
                '. Error: ' . $e->getMessage() . '. Payload: ' . json_encode($data),
                3, null, 'PrestaPicqer', $id_product
            );
            $this->sendJsonResponse(false, 'An error occurred during stock update: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Performs a full catalog stock synchronization from Picqer.
     */
    protected function syncFullCatalogStock()
    {
        // Get Picqer API details from module configuration
        $subdomain = Configuration::get(PrestaPicqer::PRESTAPICQER_API_SUBDOMAIN);
        $apiKey = Configuration::get(PrestaPicqer::PRESTAPICQER_API_KEY);

        if (empty($subdomain) || empty($apiKey)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Full sync failed. Picqer API Subdomain or Key not configured.',
                3, null, 'PrestaPicqer'
            );
            return false;
        }

        // Initialize module to access _callPicqerApi
        $module = Module::getInstanceByName('prestapicqer');
        if (!$module || !method_exists($module, '_callPicqerApi')) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Could not load module instance for full sync.',
                3, null, 'PrestaPicqer'
            );
            return false;
        }

        $offset = 0;
        $limit = 100; // Picqer pagelimit is 100
        $total_products_synced = 0;
        $missing_product_references = [];
        $items_returned_on_page = 0;

        do {
            $products_response = $module->_callPicqerApi('products?offset=' . $offset);

            if (!$products_response || !is_array($products_response)) {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Failed to fetch products from Picqer API for full sync with offset ' . $offset . '. Response: ' . json_encode($products_response),
                    3, null, 'PrestaPicqer'
                );
                break;
            }

            $products_from_picqer = $products_response;
            $items_returned_on_page = count($products_from_picqer);

            foreach ($products_from_picqer as $picqer_product) {
                $product_reference = isset($picqer_product['productcode']) ? pSQL($picqer_product['productcode']) : '';
                
                $total_freestock = 0;
                if (isset($picqer_product['stock']) && is_array($picqer_product['stock'])) {
                    foreach ($picqer_product['stock'] as $stock_entry) {
                        if (isset($stock_entry['freestock'])) {
                            $total_freestock += (int) $stock_entry['freestock'];
                        }
                    }
                }
                $new_quantity = $total_freestock;

                $product_info = $this->getProductIdAndAttributeIdByReference($product_reference);

                if (!$product_info || empty($product_info['id_product'])) {
                    $missing_product_references[] = $product_reference;
                    continue;
                }

                $id_product = (int) $product_info['id_product'];
                $id_product_attribute = (int) $product_info['id_product_attribute'];
                $id_shop = Context::getContext()->shop->id;

                try {
                    $old_quantity = Product::getQuantity($id_product, $id_product_attribute);
                    StockAvailable::setQuantity($id_product, $id_product_attribute, $new_quantity, $id_shop);
                    $updated_quantity = Product::getQuantity($id_product, $id_product_attribute);

                    if ($updated_quantity == $new_quantity) {
                        $total_products_synced++;
                        if($new_quantity != $old_quantity){
                            PrestaShopLogger::addLog(
                                'PrestaPicqer Webhook: Full sync: Stock updated for Product ID ' . $id_product .
                                ($id_product_attribute ? ' (Attribute ID ' . $id_product_attribute . ')' : '') .
                                ' from ' . $old_quantity . ' to ' . $updated_quantity . ' units.',
                                1, null, 'PrestaPicqer', $id_product
                            );
                        }
                    } else {
                        PrestaShopLogger::addLog(
                            'PrestaPicqer Webhook: Full sync: Failed to update stock for Product ID ' . $id_product .
                            '. Target: ' . $new_quantity . ', Actual: ' . $updated_quantity . '.',
                            3, null, 'PrestaPicqer', $id_product
                        );
                    }
                } catch (Exception $e) {
                    PrestaShopLogger::addLog(
                        'PrestaPicqer Webhook: Full sync: Exception during stock update for Product ID ' . $id_product .
                        '. Error: ' . $e->getMessage(),
                        3, null, 'PrestaPicqer', $id_product
                    );
                }
            }
            $offset += $limit;
        } while ($items_returned_on_page === $limit);

        // Log all missing product references
        if (!empty($missing_product_references)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Full sync: Product SKUs not found by reference in PrestaShop: ' . implode(', ', $missing_product_references),
                2, // Warning level
                null,
                'PrestaPicqer'
            );
        }

        PrestaShopLogger::addLog(
            'PrestaPicqer Webhook: Full catalog stock sync completed. Total products updated: ' . $total_products_synced,
            1, null, 'PrestaPicqer'
        );
        return true;
    }

    /**
     * Processes an order status changed webhook from Picqer.
     */
    protected function processOrderStatusWebhook($data)
    {
        $pull_order_status_enabled = (bool) Configuration::get(PrestaPicqer::PRESTAPICQER_PULL_ORDER_STATUS_ENABLED);

        if (!$pull_order_status_enabled) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Pull Order Status is disabled. Skipping order status update.',
                1, null, 'PrestaPicqer'
            );
            $this->sendJsonResponse(true, 'Pull Order Status is disabled.', 200);
            return;
        }

        if (!isset($data['data']) || !isset($data['data']['reference']) || !isset($data['data']['status'])) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Missing order data, order reference, or status in order status update payload. Payload: ' . json_encode($data),
                3, // Error
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(false, 'Missing order data, order reference, or status.', 200);
            return;
        }

        $picqer_order_reference = pSQL($data['data']['reference']);
        $picqer_status = Tools::strtolower(pSQL($data['data']['status']));
        $shop_prefix = Configuration::get(PrestaPicqer::PRESTAPICQER_SHOP_IDENTIFIER);

        $matches = [];
        if (preg_match('/^(.{1,3})\s#(\d+)\s\/\s(.*)$/', $picqer_order_reference, $matches)){
            if($shop_prefix === $matches[1]){
                $id_order = intval($matches[2]);
                if($id_order > 0){
                    $order = new Order($id_order);
                    if($order->reference != $matches[3]){
                        unset($order);
                    }
                }
            }
        }
       
        if (!isset($order)){
            if($shop_prefix === $matches[1]) {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: PrestaShop Order not found for reference: ' . $picqer_order_reference,
                    2, // Warning
                    null,
                    'PrestaPicqer'
                );
                $this->sendJsonResponse(false, 'PrestaShop Order not found.', 404);
                return;
            }
            else {
                $this->sendJsonResponse(true, 'Order is not for this PrestaShop.', 200);
            }
        }


        $picqer_tracking = [];
        //check for Picqer trackingcodes on this order
        if(isset($data['data']['picklists']) && is_array($data['data']['picklists']) && !empty($data['data']['picklists'])){

            $module = Module::getInstanceByName('prestapicqer');
            if (!$module || !method_exists($module, '_callPicqerApi')) {
                PrestaShopLogger::addLog(
                    'PrestaPicqer Webhook: Could not load module instance for order status process.',
                    3, null, 'PrestaPicqer'
                );
                return;
            }

            foreach($data['data']['picklists'] as $picklist){
                $shipments = $module->_callPicqerApi('picklists/'.$picklist['idpicklist'].'/shipments');
                if(isset($shipments) && is_array($shipments)){
                    foreach($shipments as $shipment){
                        if(isset($shipment['trackingcode']) && !empty($shipment['trackingcode'])){
                            $picqer_tracking[] = [
                                'code' => $shipment['trackingcode'],
                                'url' => !empty($shipment['tracktraceurl']) ? $shipment['tracktraceurl'] : '' //trackingurl
                            ];
                        }
                    }
                }
            }
        }

        // Get status mappings from DB
        $status_mappings = Db::getInstance()->executeS('SELECT * FROM `' . _DB_PREFIX_ . 'prestapicqer_status_map`');

        $new_ps_order_state_id = 0;
        foreach ($status_mappings as $mapping) {
            // Check for 'Any' PS from status OR specific PS from status
            if (($mapping['ps_from_status_id'] == 0 || $mapping['ps_from_status_id'] == $order->current_state) &&
                $mapping['picqer_status_name'] == ucfirst($picqer_status)) {
                $new_ps_order_state_id = (int) $mapping['ps_to_status_id'];
                break; 
            }
        }

        if ($new_ps_order_state_id === 0) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: No matching status mapping found for Picqer status "' . $picqer_status . '" from PS status ' . $order->current_state . '. Order Ref: ' . $picqer_order_reference,
                2, // Warning
                null,
                'PrestaPicqer'
            );
            $this->sendJsonResponse(true, 'No matching status mapping found.', 200);
            return;
        }

        // Check if the order status actually needs to be changed
        if ($order->current_state == $new_ps_order_state_id) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Order ' . $order->reference . ' already in target status ' . $picqer_status . '. No update needed.',
                1, // Info
                null,
                'PrestaPicqer',
                $order->id
            );
            $this->sendJsonResponse(true, 'Order status already up-to-date.', 200);
            return;
        }

        try {
            // Create a new OrderHistory entry to change the status
            $history = new OrderHistory();
            $history->id_order = $order->id;
            $history->id_employee = 0; // Use 0 for automated updates
            $history->changeIdOrderState($new_ps_order_state_id, $order, true);
            
            if(!empty($picqer_tracking)){

                $new_shipping_number = implode(', ', array_column($picqer_tracking, 'code'));
                $current_shippingnumber = $order->getShippingNumber();
                if($current_shippingnumber !== $new_shipping_number){
                    $id_order_carrier = $order->getIdOrderCarrier();
                    if ($id_order_carrier) {
                        $orderCarrier = new OrderCarrier($id_order_carrier);
                        $orderCarrier->tracking_number = $new_shipping_number;
                        $orderCarrier->save();   // or update()
                    }
                }
            }

            $history->addWithemail(true); // Send email to customer if configured for this status
            // $history->add();

            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Order ' . $order->reference . ' status updated to ' . $picqer_status . ' (PS State ID: ' . $new_ps_order_state_id . ').',
                1, // Info
                null,
                'PrestaPicqer',
                $order->id
            );
            $this->sendJsonResponse(true, 'Order status updated successfully.', 200);
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                'PrestaPicqer Webhook: Exception during order status update for Order ' . $order->reference . '. Error: ' . $e->getMessage() . '. Payload: ' . json_encode($data),
                3, // Error
                null,
                'PrestaPicqer',
                $order->id
            );
            $this->sendJsonResponse(false, 'An error occurred during order status update: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Helper method to find product and attribute ID by reference.
     * This replaces the non-existent Product::getProductIdAndAttributeIdByReference().
     *
     * @param string $reference The product reference (SKU).
     * @return array|false An array with 'id_product' and 'id_product_attribute', or false if not found.
     */
    protected function getProductIdAndAttributeIdByReference($reference)
    {
        if (empty($reference)) {
            return false;
        }

        // First, try to find a product with this reference (for simple products)
        $sql = 'SELECT `id_product` FROM `' . _DB_PREFIX_ . 'product` WHERE `reference` = \'' . pSQL($reference) . '\'';
        $id_product = (int) Db::getInstance()->getValue($sql);

        if ($id_product > 0) {
            return [
                'id_product' => $id_product,
                'id_product_attribute' => 0, // Simple product
            ];
        }

        // If not found as a simple product, try to find it as a product attribute
        $sql = 'SELECT pa.`id_product`, pa.`id_product_attribute`
                FROM `' . _DB_PREFIX_ . 'product_attribute` pa
                WHERE pa.`reference` = \'' . pSQL($reference) . '\'';
        $result = Db::getInstance()->getRow($sql);

        if ($result && !empty($result['id_product'])) {
            return [
                'id_product' => (int) $result['id_product'],
                'id_product_attribute' => (int) $result['id_product_attribute'],
            ];
        }

        return false;
    }

    /**
     * Helper method to send JSON response and terminate script execution.
     *
     * @param bool $success   Indicates if the operation was successful.
     * @param string $message A message describing the outcome.
     * @param int $http_code The HTTP status code to send with the response.
     */
    protected function sendJsonResponse($success, $message, $http_code)
    {
        http_response_code($http_code);
        echo json_encode([
            'success' => $success,
            'message' => $message,
        ]);
        exit;
    }
}