<?php

/**
 * PrestaPicqer - PrestaShop Module
 * 
 * @author     SofTer B.V.
 * @copyright  2025 SofTer B.V.
 * @license    Commercial License - See license.txt
 * @version    1.0.8
 * 
 * LICENSE: This module is licensed for use on a single PrestaShop installation.
 * Redistribution, resale, or modification without permission is strictly prohibited.
 * 
 * For support: softernl@protonmail.com
 * For license terms: See license.txt
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

use Sendcloud\PrestaShop\ServicePoint\ServicePointRepository;

class PrestaPicqer extends Module
{
    // General Settings
    const PRESTAPICQER_API_SUBDOMAIN = 'PRESTAPICQER_API_SUBDOMAIN';
    const PRESTAPICQER_API_KEY = 'PRESTAPICQER_API_KEY';
    const PRESTAPICQER_WEBHOOK_SECRET = 'PRESTAPICQER_WEBHOOK_SECRET';

    // Order Settings (PrestaShop to Picqer & Picqer to PrestaShop Order Status)
    const PRESTAPICQER_ORDER_SYNC_ENABLED = 'PRESTAPICQER_ORDER_SYNC_ENABLED';
    const PRESTAPICQER_ORDER_CREATE_STATUSES = 'PRESTAPICQER_ORDER_CREATE_STATUSES'; // Comma-separated IDs
    const PRESTAPICQER_SHOP_IDENTIFIER = 'PRESTAPICQER_SHOP_IDENTIFIER';
    const PRESTAPICQER_USE_PS_PRODUCT_PRICE = 'PRESTAPICQER_USE_PS_PRODUCT_PRICE';
    const PRESTAPICQER_CREATE_MISSING_PRODUCTS = 'PRESTAPICQER_CREATE_MISSING_PRODUCTS';
    const PRESTAPICQER_ALLOCATE_STOCK = 'PRESTAPICQER_ALLOCATE_STOCK';
    const PRESTAPICQER_ORDER_PROCESS_STATUSES = 'PRESTAPICQER_ORDER_PROCESS_STATUSES'; // Comma-separated IDs
    const PRESTAPICQER_ORDER_CANCEL_STATUSES = 'PRESTAPICQER_ORDER_CANCEL_STATUSES'; 
    const PRESTAPICQER_PULL_ORDER_STATUS_ENABLED = 'PRESTAPICQER_PULL_ORDER_STATUS_ENABLED';
    const PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID = 'PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID'; // Picqer's webhook ID
    const PRESTAPICQER_ORDER_TAGS = 'PRESTAPICQER_ORDER_TAGS'; // Comma-separated IDs of Picqer tags
    const PRESTAPICQER_CREATE_CUSTOMERS_ENABLED = 'PRESTAPICQER_CREATE_CUSTOMERS_ENABLED'; 

    // Stock Settings (Picqer to PrestaShop Stock)
    const PRESTAPICQER_PULL_STOCK_ENABLED = 'PRESTAPICQER_PULL_STOCK_ENABLED';
    const PRESTAPICQER_PULL_STOCK_FULL_SYNC = 'PRESTAPICQER_PULL_STOCK_FULL_SYNC';
    const PRESTAPICQER_STOCK_WEBHOOK_ID = 'PRESTAPICQER_STOCK_WEBHOOK_ID'; // Picqer's webhook ID
    const PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID = 'PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID';

    const DEBUG_LOGGING = false;

    protected $picqerStatusKeys = ['Concept', 'Expected', 'Processing', 'Paused', 'Completed', 'Cancelled'];

    public function __construct()
    {
        $this->name = 'prestapicqer';
        $this->tab = 'administration';
        $this->version = '1.0.8';
        $this->author = 'SofTer';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.7.0.0',
            'max' => _PS_VERSION_,
        ];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Picqer API Integration Module');
        $this->description = $this->l('Synchronizes orders to Picqer and receives stock/status updates via webhook.');

        $this->confirmUninstall = $this->l('Are you sure you want to uninstall? All configuration data and status mappings will be lost.');

        $this->l('Concept', 'prestapicqer');
        $this->l('Expected', 'prestapicqer');
        $this->l('Processing', 'prestapicqer');
        $this->l('Paused', 'prestapicqer');
        $this->l('Completed', 'prestapicqer');
        $this->l('Cancelled', 'prestapicqer');
        // -----------
        // $order = new Order(5);
        // $this->addOrderMessage($order, 'Hallo kees');
        // echo 'hallo';
        // echo $order->total_shipping_tax_excl;
        // echo $order->payment;	
        // $carrier = new Carrier((int)$order->id_carrier);


        // $delivery_address = new Address($order->id_address_delivery);
        // $orderCarrier = new OrderCarrier($order->id_carrier);

        // $orderCarriers = OrderCarrier::getOrderCarriers((int) $order->id);
        // foreach ($orderCarriers as $oc) {
        //     echo $oc['tracking_number'];
        // }

        // $idOrderCarrier = (int) $order->getIdOrderCarrier();
        // $orderCarrier   = new OrderCarrier($idOrderCarrier);


        // ServicePointRepository
        // $sendcloudModule = Module::getInstanceByName('sendcloudv2');
        // dump($sendcloudModule);

        // $cart = new Cart($order->id_cart);
        // $servicePoint = $sendcloudModule->hookDisplayBeforeCarrier(['cart' => $cart]);
        // dump($servicePoint);
        // dump($sendcloudModule->smarty);

        // $smartyTpl = $sendcloudModule->smarty->tpl_vars;
        // if(isset($smartyTpl['service_point_details'])){
        //     $servicePointDataJson = $smartyTpl['service_point_details'];
        //     dump(json_decode($servicePointDataJson, true));
        // }
        // dump($smartyTpl['service_point_details']);
        // // $servicePoint = $sendcloudModule->getMessage('service_point_details');
        // $smarty = $smarty;
        // dump($smarty);
        // foreach($smarty->_tpl_vars as $key=>$value) {
        //     echo "$key is $value<br>";
        // }
        // dump($servicePoint);

        // $repo = $sendcloudModule->get('sendcloud.prestashop.service_point_repository');
        // $repo = $sendcloudModule->get('sendcloud.prestashop.ServicePointRepository');
        // dump($repo);
        // $servicePoint = $repo->findByCartId($id_cart);

        // if ($servicePoint) {
        //     // $servicePoint is an associative array or DTO that contains
        //     //  - external_id      (Sendcloud point id)
        //     //  - name             (the shop name)
        //     //  - street, city, postal_code, country
        //     //  - carrier_code
        //     //  - … whatever else the module stores
        //     var_dump($servicePoint);
        // }
        


        // $id_cart = (int) $order->id_cart

        // $row = Db::getInstance()->getRow('
        //     SELECT *
        //     FROM `'._DB_PREFIX_ .'service_points_v2`
        //     WHERE `id_cart` = ' . $id_cart
        // );

        // if ($row) {
        //     // $row contains the same fields mentioned earlier
        //     dump($row);
        // }





        // dump($orderCarrier);
        // $cart = new Cart($order->id_cart)
        // $cartId = $order->id_cart;
        // echo $cartId;
        // echo $carrier->getServicePointService()->getByCartId($cartId);
        // $product = new Product(19);
        // $product->loadStockData();
        // echo $product->quantity;


        //echo 'Hallo'.Configuration::get(self::PRESTAPICQER_WEBHOOK_SECRET);
        //echo 'hhh'.Tools::strtoupper(Tools::passwdGen(32));
    }



    /**
     * This method is executed when the module is installed.
     */
    public function install()
    {
        if (!parent::install() ||
            !$this->registerHook('actionValidateOrder') || // For new orders
            !$this->registerHook('actionOrderStatusUpdate') || // For order status changes
            !$this->_installDb() // Install custom database table
            
        ) {
            return false;
        }

        // Set default values for configuration
        Configuration::updateValue(self::PRESTAPICQER_API_SUBDOMAIN, '');
        Configuration::updateValue(self::PRESTAPICQER_API_KEY, '');
        // Generate and store the webhook secret only on install
        Configuration::updateValue(self::PRESTAPICQER_WEBHOOK_SECRET, Tools::strtoupper(Tools::passwdGen(32)));

        Configuration::updateValue(self::PRESTAPICQER_ORDER_SYNC_ENABLED, false); // Default to disabled
        Configuration::updateValue(self::PRESTAPICQER_ORDER_CREATE_STATUSES, ''); // No default statuses
        Configuration::updateValue(self::PRESTAPICQER_SHOP_IDENTIFIER, 'PS'); // Default shop identifier
        Configuration::updateValue(self::PRESTAPICQER_USE_PS_PRODUCT_PRICE, true); // Default to use PS price
        Configuration::updateValue(self::PRESTAPICQER_CREATE_MISSING_PRODUCTS, true); // Default create missing products
        Configuration::updateValue(self::PRESTAPICQER_ALLOCATE_STOCK, false); // Default to allocate stock
        Configuration::updateValue(self::PRESTAPICQER_ORDER_PROCESS_STATUSES, ''); // No default statuses
        Configuration::updateValue(self::PRESTAPICQER_ORDER_CANCEL_STATUSES, ''); // No default statuses
        Configuration::updateValue(self::PRESTAPICQER_ORDER_TAGS, ''); // No default tags
        Configuration::updateValue(self::PRESTAPICQER_CREATE_CUSTOMERS_ENABLED, false);

        Configuration::updateValue(self::PRESTAPICQER_PULL_STOCK_ENABLED, false); // Default to disabled
        Configuration::updateValue(self::PRESTAPICQER_PULL_STOCK_FULL_SYNC, true); // Default to full catalog sync
        Configuration::updateValue(self::PRESTAPICQER_PULL_ORDER_STATUS_ENABLED, false); // Default to disabled
        Configuration::updateValue(self::PRESTAPICQER_STOCK_WEBHOOK_ID, 0); // No webhook ID initially
        Configuration::updateValue(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID, 0);
        Configuration::updateValue(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID, 0); // No webhook ID initially

       

        return true;
    }

    /**
     * This method is executed when the module is uninstalled.
     */
    public function uninstall()
    {
        if (!parent::uninstall() ||
            !$this->unregisterHook('actionValidateOrder') ||
            !$this->unregisterHook('actionOrderStatusUpdate') ||
            !$this->_uninstallDb() // Uninstall custom database table
        ) {
            return false;
        }

        // Delete configuration values
        Configuration::deleteByName(self::PRESTAPICQER_API_SUBDOMAIN);
        Configuration::deleteByName(self::PRESTAPICQER_API_KEY);
        Configuration::deleteByName(self::PRESTAPICQER_WEBHOOK_SECRET);
        Configuration::deleteByName(self::PRESTAPICQER_ORDER_SYNC_ENABLED);
        Configuration::deleteByName(self::PRESTAPICQER_ORDER_CREATE_STATUSES);
        Configuration::deleteByName(self::PRESTAPICQER_SHOP_IDENTIFIER);
        Configuration::deleteByName(self::PRESTAPICQER_USE_PS_PRODUCT_PRICE);
        Configuration::deleteByName(self::PRESTAPICQER_CREATE_MISSING_PRODUCTS);
        Configuration::deleteByName(self::PRESTAPICQER_ALLOCATE_STOCK);
        Configuration::deleteByName(self::PRESTAPICQER_ORDER_PROCESS_STATUSES);
        Configuration::deleteByName(self::PRESTAPICQER_ORDER_CANCEL_STATUSES);
        Configuration::deleteByName(self::PRESTAPICQER_ORDER_TAGS);
        Configuration::deleteByName(self::PRESTAPICQER_CREATE_CUSTOMERS_ENABLED);

        Configuration::deleteByName(self::PRESTAPICQER_PULL_STOCK_ENABLED);
        Configuration::deleteByName(self::PRESTAPICQER_PULL_STOCK_FULL_SYNC);
        Configuration::deleteByName(self::PRESTAPICQER_PULL_ORDER_STATUS_ENABLED);
        Configuration::deleteByName(self::PRESTAPICQER_STOCK_WEBHOOK_ID);
        Configuration::deleteByName(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID);
        Configuration::deleteByName(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID);

        return true;
    }

    /**
     * Installs the custom database table for status mappings.
     */
    protected function _installDb()
    {
        $sql = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'prestapicqer_status_map` (
            `id_prestapicqer_status_map` INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,
            `ps_from_status_id` INT(10) UNSIGNED NOT NULL,
            `picqer_status_name` VARCHAR(255) NOT NULL,
            `ps_to_status_id` INT(10) UNSIGNED NOT NULL,
            PRIMARY KEY (`id_prestapicqer_status_map`)
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8;';

        return Db::getInstance()->execute($sql);
    }

    /**
     * Uninstalls the custom database table.
     */
    protected function _uninstallDb()
    {
        $sql = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'prestapicqer_status_map`;';
        return Db::getInstance()->execute($sql);
    }

    /**
     * This method handles the module's configuration page.
     */
    public function getContent()
    {
        $output = '';

        // Sync webhook IDs from Picqer API on page load to ensure accurate display
        $this->_syncWebhookIdsFromPicqer();

        // Handle form submissions
        if (Tools::isSubmit('submitPrestaPicqerGeneral')) {
            $output .= $this->processGeneralSettings();
        } elseif (Tools::isSubmit('submitPrestaPicqerOrders')) { 
            $output .= $this->processOrderSettings();
        } elseif (Tools::isSubmit('submitPrestaPicqerStock')) {
            $output .= $this->processStockSettings();
        } elseif (Tools::isSubmit('createStockWebhooks')) {
            $output .= $this->createBothStockWebhooks();
        } elseif (Tools::isSubmit('deleteStockWebhooks')) {
            $output .= $this->deleteBothStockWebhooks();
        } elseif (Tools::isSubmit('deleteStockWebhook')) {
            $output .= $this->deletePicqerWebhook(
                Configuration::get(self::PRESTAPICQER_STOCK_WEBHOOK_ID),
                self::PRESTAPICQER_STOCK_WEBHOOK_ID
            );
        } elseif (Tools::isSubmit('createOrderStatusWebhook')) {
            $output .= $this->createPicqerWebhook(
                'orders.status_changed',
                self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID
            );
        } elseif (Tools::isSubmit('deleteOrderStatusWebhook')) {
            $output .= $this->deletePicqerWebhook(
                Configuration::get(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID),
                self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID
            );
        }

        // Render the configuration form using a custom template
        return $output . $this->display(__FILE__, 'views/templates/admin/configure.tpl');
    }

    /**
     * Process general settings form submission.
     */
    protected function processGeneralSettings()
    {
        $subdomain = (string) Tools::getValue(self::PRESTAPICQER_API_SUBDOMAIN);
        $apiKey = (string) Tools::getValue(self::PRESTAPICQER_API_KEY);
        $shopIdentifier = (string) Tools::getValue(self::PRESTAPICQER_SHOP_IDENTIFIER);

        if (empty($subdomain) || empty($apiKey)) {
            return $this->displayError($this->l('Picqer API Subdomain and API Key are required.'));
        }

        // Validate shop identifier
        if (!preg_match('/^[a-zA-Z0-9]{1,3}$/', $shopIdentifier)) {
            return $this->displayError($this->l('Shop Identifier must be 1-3 alphanumeric characters.'));
        }

        Configuration::updateValue(self::PRESTAPICQER_API_SUBDOMAIN, $subdomain);
        Configuration::updateValue(self::PRESTAPICQER_API_KEY, $apiKey);
        Configuration::updateValue(self::PRESTAPICQER_SHOP_IDENTIFIER, Tools::strtoupper($shopIdentifier));

        return $this->displayConfirmation($this->l('General settings updated successfully!'));
    }

    /**
     * Process all order-related settings (PrestaShop to Picqer & Picqer to PrestaShop Order Status).
     */
    protected function processOrderSettings()
    {
        $output = '';

        // --- Process PrestaShop to Picqer Order Settings ---
        $orderSyncEnabled = (bool) Tools::getValue(self::PRESTAPICQER_ORDER_SYNC_ENABLED);
        $orderCreateStatuses = Tools::getValue(self::PRESTAPICQER_ORDER_CREATE_STATUSES);
        $usePsProductPrice = (bool) Tools::getValue(self::PRESTAPICQER_USE_PS_PRODUCT_PRICE);
        $createMissingProducts = (bool) Tools::getValue(self::PRESTAPICQER_CREATE_MISSING_PRODUCTS);
        $allocateStock = (bool) Tools::getValue(self::PRESTAPICQER_ALLOCATE_STOCK);
        $orderProcessStatuses = Tools::getValue(self::PRESTAPICQER_ORDER_PROCESS_STATUSES);
        $orderCancelStatuses = Tools::getValue(self::PRESTAPICQER_ORDER_CANCEL_STATUSES);
        $orderTags = Tools::getValue(self::PRESTAPICQER_ORDER_TAGS);

        Configuration::updateValue(self::PRESTAPICQER_ORDER_SYNC_ENABLED, $orderSyncEnabled);
        Configuration::updateValue(self::PRESTAPICQER_ORDER_CREATE_STATUSES, implode(',', array_map('intval', (array)$orderCreateStatuses)));
        Configuration::updateValue(self::PRESTAPICQER_USE_PS_PRODUCT_PRICE, $usePsProductPrice);
        Configuration::updateValue(self::PRESTAPICQER_CREATE_MISSING_PRODUCTS, $createMissingProducts);
        Configuration::updateValue(self::PRESTAPICQER_ALLOCATE_STOCK, $allocateStock);
        Configuration::updateValue(self::PRESTAPICQER_ORDER_PROCESS_STATUSES, implode(',', array_map('intval', (array)$orderProcessStatuses)));
        Configuration::updateValue(self::PRESTAPICQER_ORDER_CANCEL_STATUSES, implode(',', array_map('intval', (array)$orderCancelStatuses)));
        Configuration::updateValue(self::PRESTAPICQER_ORDER_TAGS, implode(',', array_map('intval', (array)$orderTags)));

        
        // --- Process Picqer to PrestaShop Order Status Settings ---
        $pullOrderStatusEnabled = (bool) Tools::getValue(self::PRESTAPICQER_PULL_ORDER_STATUS_ENABLED);
        Configuration::updateValue(self::PRESTAPICQER_PULL_ORDER_STATUS_ENABLED, $pullOrderStatusEnabled);

        

        // --- Process Status Map Settings ---
        $db = Db::getInstance();
        $table_name = 'prestapicqer_status_map';

        // Clear existing mappings
        $db->execute('TRUNCATE TABLE `' . _DB_PREFIX_ .$table_name . '`;');

        $psFromStatusIds = Tools::getValue('ps_from_status_id');
        $picqerStatusNames = Tools::getValue('picqer_status_name');
        $psToStatusIds = Tools::getValue('ps_to_status_id');

        if (is_array($psFromStatusIds) && count($psFromStatusIds) > 0) {
            $insert_data = [];
            for ($i = 0; $i < count($psFromStatusIds); $i++) {
                $ps_from_id = (int) $psFromStatusIds[$i];
                $picqer_name = pSQL($picqerStatusNames[$i]);
                $ps_to_id = (int) $psToStatusIds[$i];

                if ($ps_from_id >= 0 && !empty($picqer_name) && $ps_to_id > 0) {
                    $insert_data[] = [
                        'ps_from_status_id' => $ps_from_id,
                        'picqer_status_name' => $picqer_name,
                        'ps_to_status_id' => $ps_to_id,
                    ];
                }
            }

            if (!empty($insert_data)) {
                $db->insert($table_name, $insert_data);
            }
        }

        // --- Process Customer Creation Settings ---
        $createCustomersEnabled = (bool) Tools::getValue(self::PRESTAPICQER_CREATE_CUSTOMERS_ENABLED);
        Configuration::updateValue(self::PRESTAPICQER_CREATE_CUSTOMERS_ENABLED, $createCustomersEnabled);

        $output .= $this->displayConfirmation($this->l('PrestaShop to Picqer order settings updated successfully!'));

        return $output;
    }

    /**
     * Process all stock-related settings (Picqer to PrestaShop Stock).
     */
    protected function processStockSettings()
    {
        $pullStockEnabled = (bool) Tools::getValue(self::PRESTAPICQER_PULL_STOCK_ENABLED);
        $pullStockFullSync = (bool) Tools::getValue(self::PRESTAPICQER_PULL_STOCK_FULL_SYNC);

        Configuration::updateValue(self::PRESTAPICQER_PULL_STOCK_ENABLED, $pullStockEnabled);
        Configuration::updateValue(self::PRESTAPICQER_PULL_STOCK_FULL_SYNC, $pullStockFullSync);

        return $this->displayConfirmation($this->l('Picqer to PrestaShop stock settings updated successfully!'));
    }

    /**
     * Assigns configuration values and other data to the Smarty template.
     *
     * @param string $file
     * @param string $template
     * @param string|null $cache_id
     * @param string|null $compile_id
     * @return string
     */
    public function display($file, $template, $cache_id = null, $compile_id = null)
    {
        // Fetch Picqer tags for the selector
        $picqer_tags = $this->_callPicqerApi('tags');
        if (!is_array($picqer_tags)) {
            $picqer_tags = [];
            PrestaShopLogger::addLog(
                'PrestaPicqer: Failed to fetch Picqer tags from API. Check API settings.',
                2, null, 'PrestaPicqer'
            );
        }

        $translatedPicqerStatuses = [];
        foreach ($this->picqerStatusKeys as $key) {
            $translatedPicqerStatuses[$key] = $this->l($key, 'prestapicqer');
        }

        $this->context->smarty->assign([
            'lang_iso' => $this->context->language->iso_code,
            'module_dir' => $this->_path,
            'current_index' => AdminController::$currentIndex,
            'token' => Tools::getAdminTokenLite('AdminModules'),
            'ps_order_statuses' => OrderState::getOrderStates((int) $this->context->language->id),
            'webhook_url' => Context::getContext()->link->getModuleLink(
                $this->name,
                'webhook',
                ['fc' => 'module'],
                true, // Use SSL
                1, // Context::getContext()->language->id, //Always en
                Context::getContext()->shop->id
            ),
            'stock_sync_url' => Context::getContext()->link->getModuleLink(
                $this->name,
                'webhook',
                ['fc' => 'module', 'action' => 'synchronise_stock'],
                true, // Use SSL
                1, // Context::getContext()->language->id, //Always en
                Context::getContext()->shop->id
            ),
            // General Settings
            'picqer_api_subdomain' => Configuration::get(self::PRESTAPICQER_API_SUBDOMAIN),
            'picqer_api_key' => Configuration::get(self::PRESTAPICQER_API_KEY),
            'shop_identifier' => Configuration::get(self::PRESTAPICQER_SHOP_IDENTIFIER),

            // Order Settings
            'order_sync_enabled' => (bool) Configuration::get(self::PRESTAPICQER_ORDER_SYNC_ENABLED),
            'order_create_statuses' => array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_CREATE_STATUSES)))),
            'use_ps_product_price' => (bool) Configuration::get(self::PRESTAPICQER_USE_PS_PRODUCT_PRICE),
            'create_missing_products' => (bool) Configuration::get(self::PRESTAPICQER_CREATE_MISSING_PRODUCTS),
            'allocate_stock' => (bool) Configuration::get(self::PRESTAPICQER_ALLOCATE_STOCK),
            'order_process_statuses' => array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_PROCESS_STATUSES)))),
            'order_cancel_statuses' => array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_CANCEL_STATUSES)))), 
            'pull_order_status_enabled' => (bool) Configuration::get(self::PRESTAPICQER_PULL_ORDER_STATUS_ENABLED),
            'order_status_webhook_id' => Configuration::get(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID),
            'status_mappings' => $this->getStatusMappings(),
            'picqer_tags' => $picqer_tags,
            'order_tags_selected' => array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_TAGS)))),
            'create_customers_enabled' => (bool) Configuration::get(self::PRESTAPICQER_CREATE_CUSTOMERS_ENABLED),
            'picqer_statuses' => $this->picqerStatusKeys,
            'picqer_statuses_translated' => $translatedPicqerStatuses,

            // Stock Settings
            'pull_stock_enabled' => (bool) Configuration::get(self::PRESTAPICQER_PULL_STOCK_ENABLED),
            'pull_stock_full_sync' => (bool) Configuration::get(self::PRESTAPICQER_PULL_STOCK_FULL_SYNC),
            'stock_webhook_id' => Configuration::get(self::PRESTAPICQER_STOCK_WEBHOOK_ID),
            'assembled_stock_webhook_id' => Configuration::get(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID),
            
        ]);

        return parent::display($file, $template, $cache_id, $compile_id);
    }

    /**
     * Retrieves all status mappings from the database.
     */
    protected function getStatusMappings()
    {
        $db = Db::getInstance();
        $sql = 'SELECT * FROM `' . _DB_PREFIX_ . 'prestapicqer_status_map` ORDER BY id_prestapicqer_status_map ASC';
        return $db->executeS($sql);
    }


    /**
     * Makes a call to the Picqer API with automated retry for 429 Too Many Requests.
     * Also retries any other non-2xx HTTP response (e.g., 500) up to 2 extra times with a 1 second delay.
     *
     * @param string $endpoint_path The API endpoint path (e.g., 'products').
     * @param string $method The HTTP method (GET, POST, PUT, DELETE).
     * @param array $data The data to send for POST/PUT requests.
     * @param int $max_retries Maximum number of retries for 429 responses (default: 3).
     * @param int $initial_retry_delay Initial delay in seconds for the first retry (default: 5).
     * @return array|false Decoded API response on success, false on failure.
     */
    public function _callPicqerApi($endpoint_path, $method = 'GET', $data = [], $max_retries = 3, $initial_retry_delay = 5)
    {
        $subdomain = Configuration::get(self::PRESTAPICQER_API_SUBDOMAIN);
        $apiKey = Configuration::get(self::PRESTAPICQER_API_KEY);

        if (empty($subdomain) || empty($apiKey)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Picqer API Subdomain or Key not configured for API call to ' . $endpoint_path,
                3, null, 'PrestaPicqer'
            );
            return false;
        }

        $url = 'https://' . $subdomain . '.picqer.com/api/v1/' . ltrim($endpoint_path, '/');

        $headers = [
            'Content-Type: application/json',
        ];

        $retry_attempt = 0; // for 429 logic
        // Added: generic error retry controls (non-2xx, excluding 429)
        $error_retry_attempt = 0;
        $error_max_retries = 2;
        $error_retry_delay = 1;

        while ($retry_attempt <= $max_retries) {
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_HEADER, true);

            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_USERPWD, $apiKey . ':');

            if ($method === 'POST') {
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            } elseif ($method === 'PUT') {
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            } elseif ($method === 'DELETE') {
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
            }

            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);

            // Close cURL handle immediately after execution and info retrieval
            curl_close($ch);

            if ($response === false) {
                PrestaShopLogger::addLog(
                    'PrestaPicqer: Picqer API call to ' . $url . ' failed. cURL Error: ' . $curl_error,
                    3, null, 'PrestaPicqer'
                );
                return false; // No retry for cURL errors (e.g., network issues, DNS failure)
            }

            $headers_string = substr($response, 0, $header_size);
            $body_string = substr($response, $header_size);

            if ($http_code === 429) {
                $parsed_headers = $this->_parseHttpHeaders($headers_string);
                $retry_after = 0;

                // Extract X-RateLimit headers for logging
                $rate_limit_limit = isset($parsed_headers['X-RateLimit-Limit']) ? $parsed_headers['X-RateLimit-Limit'] : 'N/A';
                $rate_limit_remaining = isset($parsed_headers['X-RateLimit-Remaining']) ? $parsed_headers['X-RateLimit-Remaining'] : 'N/A';

                // Check for Retry-After header
                if (isset($parsed_headers['Retry-After'])) {
                    $retry_after_value = $parsed_headers['Retry-After'];
                    if (is_numeric($retry_after_value)) {
                        $retry_after = (int)$retry_after_value;
                    } else {
                        $retry_timestamp = strtotime($retry_after_value);
                        if ($retry_timestamp !== false) {
                            $retry_after = max(0, $retry_timestamp - time());
                        }
                    }
                }

                // Fallback if Retry-After is not present or invalid, use exponential backoff
                if ($retry_after <= 0) {
                    $retry_after = $initial_retry_delay * pow(2, $retry_attempt);
                }

                PrestaShopLogger::addLog(
                    'PrestaPicqer: Picqer API call to ' . $url . ' returned HTTP 429 (Too Many Requests). ' .
                    'Rate Limit: ' . $rate_limit_limit . ', Remaining: ' . $rate_limit_remaining . '. ' .
                    'Retrying in ' . $retry_after . ' seconds. Attempt ' . ($retry_attempt + 1) . '/' . $max_retries . '.',
                    2, null, 'PrestaPicqer'
                );

                if ($retry_attempt < $max_retries) {
                    sleep($retry_after);
                    $retry_attempt++;
                    continue;
                } else {
                    // Max retries reached
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Picqer API call to ' . $url . ' failed after ' . $max_retries . ' retries due to HTTP 429.',
                        3, null, 'PrestaPicqer'
                    );
                    return false;
                }
            }

            // If not 429, or if it was successful (2xx)
            $decoded_response = json_decode($body_string, true);

            if ($http_code >= 200 && $http_code < 300) {
                return $decoded_response;
            } else {
                // Added: retry non-2xx (excluding 429) up to 2 extra times with 1s delay
                if ($error_retry_attempt < $error_max_retries) {
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Picqer API call to ' . $url . ' returned HTTP ' . $http_code . '. ' .
                        'Retrying in ' . $error_retry_delay . ' seconds. Attempt ' . ($error_retry_attempt + 1) . '/' . $error_max_retries . '.',
                        2, null, 'PrestaPicqer'
                    );
                    $error_retry_attempt++;
                    sleep($error_retry_delay);
                    continue;
                }

                PrestaShopLogger::addLog(
                    'PrestaPicqer: Picqer API call to ' . $url . ' returned HTTP ' . $http_code . '. Request: ' . json_encode($data) . '. Response: ' . $body_string,
                    3, null, 'PrestaPicqer'
                );
                return false;
            }
        }

        return false;
    }

    /**
     * Helper function to parse raw HTTP headers string into an associative array.
     *
     * @param string $header_string The raw header string from cURL.
     * @return array An associative array of headers.
     */
    private function _parseHttpHeaders($header_string)
    {
        $headers = [];
        $lines = explode("\r\n", $header_string);
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) {
                continue;
            }
            if (strpos($line, 'HTTP/') === 0) {
                $headers['Status-Line'] = $line;
                continue;
            }
            if (strpos($line, ':') !== false) {
                list($key, $value) = explode(':', $line, 2);
                $key = str_replace(' ', '-', ucwords(str_replace('-', ' ', $key)));
                $headers[$key] = trim($value);
            }
        }
        return $headers;
    }

    /**
     * Syncs the stored webhook IDs with the actual IDs from Picqer's API.
     * This ensures the module's config page accurately reflects active webhooks.
     */
    protected function _syncWebhookIdsFromPicqer()
    {
        $webhook_url = Context::getContext()->link->getModuleLink(
            $this->name,
            'webhook',
            ['fc' => 'module'],
            true,
            1,
            Context::getContext()->shop->id
        );
        $normalized_webhook_url = rtrim($webhook_url, '/');

        $picqer_webhooks = $this->_callPicqerApi('hooks');

        $current_stock_webhook_id = 0;
        $current_assembled_webhook_id = 0;
        $current_order_status_webhook_id = 0;

        if (is_array($picqer_webhooks)) {
            foreach ($picqer_webhooks as $webhook) {
                $picqer_webhook_url = isset($webhook['address']) ? rtrim($webhook['address'], '/') : '';
                $picqer_webhook_event = isset($webhook['event']) ? $webhook['event'] : '';
                $picqer_webhook_id = isset($webhook['idhook']) ? (int) $webhook['idhook'] : 0;

                if ($picqer_webhook_url === $normalized_webhook_url && $webhook['active']) {
                    if ($picqer_webhook_event === 'products.free_stock_changed') {
                        $current_stock_webhook_id = $picqer_webhook_id;
                    } elseif ($picqer_webhook_event === 'products.assembled_stock_changed') {
                        $current_assembled_webhook_id = $picqer_webhook_id;
                    } elseif ($picqer_webhook_event === 'orders.status_changed') {
                        $current_order_status_webhook_id = $picqer_webhook_id;
                    }
                }
            }
        } else {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Picqer /hooks API response was empty or invalid (not an array).',
                2, null, 'PrestaPicqer'
            );
        }

        // Update configuration values only if they differ
        if (Configuration::get(self::PRESTAPICQER_STOCK_WEBHOOK_ID) != $current_stock_webhook_id) {
            Configuration::updateValue(self::PRESTAPICQER_STOCK_WEBHOOK_ID, $current_stock_webhook_id);
            PrestaShopLogger::addLog(
                'PrestaPicqer: Synced Stock Webhook ID from Picqer to: ' . $current_stock_webhook_id,
                1, null, 'PrestaPicqer'
            );
        }
        if (Configuration::get(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID) != $current_assembled_webhook_id) {
            Configuration::updateValue(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID, $current_assembled_webhook_id);
            PrestaShopLogger::addLog(
                'PrestaPicqer: Synced Assembled Stock Webhook ID from Picqer to: ' . $current_assembled_webhook_id,
                1, null, 'PrestaPicqer'
            );
        }
        if (Configuration::get(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID) != $current_order_status_webhook_id) {
            Configuration::updateValue(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID, $current_order_status_webhook_id);
            PrestaShopLogger::addLog(
                'PrestaPicqer: Synced Order Status Webhook ID from Picqer to: ' . $current_order_status_webhook_id,
                1, null, 'PrestaPicqer'
            );
        }
    }

    /**
     * Creates a webhook in Picqer.
     * @param string $event_type e.g., 'products.free_stock_changed', 'orders.status_changed'
     * @param string $config_key_for_id The configuration key to store the Picqer webhook ID
     */
    protected function createPicqerWebhook($event_type, $config_key_for_id)
    {
        $webhook_url = Context::getContext()->link->getModuleLink(
            $this->name,
            'webhook',
            ['fc' => 'module'],
            true,
            1,// Context::getContext()->language->id, //Always en
            Context::getContext()->shop->id
        );
        $webhook_secret = Configuration::get(self::PRESTAPICQER_WEBHOOK_SECRET);
        if(empty($webhook_secret)){
            Configuration::updateValue(self::PRESTAPICQER_WEBHOOK_SECRET, Tools::strtoupper(Tools::passwdGen(32)));
            $webhook_secret = Configuration::get(self::PRESTAPICQER_WEBHOOK_SECRET);
        }
        $shopIdentifier = Configuration::get(self::PRESTAPICQER_SHOP_IDENTIFIER);

        if (empty($shopIdentifier)) {
            return $this->displayError($this->l('Shop Identifier Prefix is not configured. Please check General settings.'));
        }

        // Construct the webhook name
        $webhook_name = Tools::strtoupper($shopIdentifier) . '_' . str_replace('.', '_', $event_type);

        // Check if webhook already exists
        $this->_syncWebhookIdsFromPicqer();
        if (
            ($config_key_for_id === self::PRESTAPICQER_STOCK_WEBHOOK_ID && Configuration::get(self::PRESTAPICQER_STOCK_WEBHOOK_ID) > 0) ||
            ($config_key_for_id === self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID && Configuration::get(self::PRESTAPICQER_ORDER_STATUS_WEBHOOK_ID) > 0)
        ) {
            return $this->displayConfirmation(sprintf($this->l('Webhook for %s already exists and is registered.'), $event_type));
        }

        $data = [
            'address' => $webhook_url,
            'event' => $event_type,
            'secret' => $webhook_secret,
            'name' => $webhook_name,
        ];

        $response = $this->_callPicqerApi('hooks', 'POST', $data);

        if ($response && isset($response['idhook'])) {
            Configuration::updateValue($config_key_for_id, (int) $response['idhook']);
            return $this->displayConfirmation(sprintf($this->l('Webhook for %s created successfully!'), $event_type));
        } else {
            return $this->displayError(sprintf($this->l('Failed to create webhook for %s. Check Picqer API settings and logs.'), $event_type));
        }
    }

    /**
     * Creates both stock webhooks (free_stock and assembled_stock)
     */
    protected function createBothStockWebhooks()
    {
        
        $webhook_url = Context::getContext()->link->getModuleLink(
            $this->name,
            'webhook',
            ['fc' => 'module'],
            true,
            1,
            Context::getContext()->shop->id
        );
        $webhook_secret = Configuration::get(self::PRESTAPICQER_WEBHOOK_SECRET);
        if(empty($webhook_secret)){
            Configuration::updateValue(self::PRESTAPICQER_WEBHOOK_SECRET, Tools::strtoupper(Tools::passwdGen(32)));
            $webhook_secret = Configuration::get(self::PRESTAPICQER_WEBHOOK_SECRET);
        }
        $shopIdentifier = Configuration::get(self::PRESTAPICQER_SHOP_IDENTIFIER);

        if (empty($shopIdentifier)) {
            return $this->displayError($this->l('Shop Identifier Prefix is not configured. Please check General settings.'));
        }

        $this->_syncWebhookIdsFromPicqer();
        
        $stock_webhook_exists =  self::PRESTAPICQER_STOCK_WEBHOOK_ID && (int)Configuration::get(self::PRESTAPICQER_STOCK_WEBHOOK_ID) > 0;
        $assembled_webhook_exists = self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID && (int)Configuration::get(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID) > 0;

        if ($stock_webhook_exists && $assembled_webhook_exists) {
            return $this->displayConfirmation($this->l('Both stock webhooks already exist and are registered.'));
        }

        $output = '';
        $errors = [];

        // Create free stock webhook
        if (!$stock_webhook_exists) {
            $webhook_name = Tools::strtoupper($shopIdentifier) . '_products_free_stock_changed';
            $data = [
                'address' => $webhook_url,
                'event' => 'products.free_stock_changed',
                'secret' => $webhook_secret,
                'name' => $webhook_name,
            ];
            $response = $this->_callPicqerApi('hooks', 'POST', $data);
            if ($response && isset($response['idhook'])) {
                Configuration::updateValue(self::PRESTAPICQER_STOCK_WEBHOOK_ID, (int) $response['idhook']);
            } else {
                $errors[] = $this->l('Failed to create free stock webhook.');
            }
        }

        // Create assembled stock webhook
        if (!$assembled_webhook_exists) {
            $webhook_name = Tools::strtoupper($shopIdentifier) . '_products_assembled_stock_changed';
            $data = [
                'address' => $webhook_url,
                'event' => 'products.assembled_stock_changed',
                'secret' => $webhook_secret,
                'name' => $webhook_name,
            ];
            $response = $this->_callPicqerApi('hooks', 'POST', $data);
            if ($response && isset($response['idhook'])) {
                Configuration::updateValue(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID, (int) $response['idhook']);
            } else {
                $errors[] = $this->l('Failed to create assembled stock webhook.');
            }
        }

        if (!empty($errors)) {
            return $this->displayError(implode('<br>', $errors));
        }

        return $this->displayConfirmation($this->l('Both stock webhooks created successfully!'));
    }

    /**
     * Deletes both stock webhooks
     */
    protected function deleteBothStockWebhooks()
    {
        $stock_webhook_id = Configuration::get(self::PRESTAPICQER_STOCK_WEBHOOK_ID);
        $assembled_webhook_id = Configuration::get(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID);

        $errors = [];

        if (!empty($stock_webhook_id)) {
            $response = $this->_callPicqerApi('hooks/' . (int) $stock_webhook_id, 'DELETE');
            if (empty($response)) {
                Configuration::updateValue(self::PRESTAPICQER_STOCK_WEBHOOK_ID, 0);
            } else {
                $errors[] = $this->l('Failed to delete free stock webhook.');
            }
        }

        if (!empty($assembled_webhook_id)) {
            $response = $this->_callPicqerApi('hooks/' . (int) $assembled_webhook_id, 'DELETE');
            if (empty($response)) {
                Configuration::updateValue(self::PRESTAPICQER_ASSEMBLED_STOCK_WEBHOOK_ID, 0);
            } else {
                $errors[] = $this->l('Failed to delete assembled stock webhook.');
            }
        }

        if (!empty($errors)) {
            return $this->displayError(implode('<br>', $errors));
        }

        return $this->displayConfirmation($this->l('Both stock webhooks deactivated successfully!'));
    }

    /**
     * Deletes a webhook in Picqer.
     * @param int $webhook_id The ID of the webhook in Picqer
     * @param string $config_key_for_id The configuration key where the Picqer webhook ID is stored
     */
    protected function deletePicqerWebhook($webhook_id, $config_key_for_id)
    {
        if (empty($webhook_id)) {
            return $this->displayError($this->l('No webhook ID found to deactivate.'));
        }

        $response = $this->_callPicqerApi('hooks/' . (int) $webhook_id, 'DELETE');

        if (empty($response)) {
            Configuration::updateValue($config_key_for_id, 0);
            return $this->displayConfirmation($this->l('Webhook deactivated successfully!'));
        } else {
            return $this->displayError($this->l('Failed to deactivate webhook. Check Picqer API settings and logs.'));
        }
    }

    /**
     * Adds a message to the order
     * @param object $order The order
     * @param string $messageText The message text
     */
    protected function addOrderMessage($order, $messageText){
        $idOrder = (int)$order->id;
        $idCustomer = (int)$order->id_customer;
        $email = (string)(new Customer($idCustomer))->email;

        // 1. Create or reuse customer thread
        $threadId = (int)Db::getInstance()->getValue(
            'SELECT id_customer_thread
            FROM '._DB_PREFIX_.'customer_thread
            WHERE id_order = '.$idOrder.' AND id_shop = '.(int)$order->id_shop
        );

        if (!$threadId) {
            // Create a new customer thread if one doesn't exist for this order
            $customerThread = new CustomerThread();
            $customerThread->id_shop = (int)$order->id_shop;
            $customerThread->id_lang = (int)$order->id_lang;
            $customerThread->id_contact = 1; // Use 0 for system/module generated messages, or a specific contact ID
            $customerThread->id_order = $idOrder;
            $customerThread->id_customer = $idCustomer;
            $customerThread->email = $email;
            $customerThread->status = 'open'; // Or 'closed' if you prefer
            $customerThread->token = Tools::passwdGen(12); // Generate a unique token
            $customerThread->add();
            $threadId = (int)$customerThread->id;
        }

        // 2. Insert the message into the customer_message table
        if ($threadId) {
            $customerMessage = new CustomerMessage();
            $customerMessage->id_customer_thread = $threadId;
            $customerMessage->id_employee = 1; // 0 for module-generated, or an employee ID
            $customerMessage->message = pSQL($messageText);
            $customerMessage->private = 1; // 1 for private (back office only), 0 for public (visible to customer)
            $customerMessage->add();
        }
    }

    /**
     * Returns the cover image of an OrderDetail row as a base-64 data-URI.
     *
     * @param array $orderDetailRow   One row from $order->getProducts()
     * @param string $imageFormat     'jpg' or 'png'  (jpg is ~30 % smaller)
     * @return string                 data:image/...;base64,xxxxxxxx
     */
    protected function getOrderDetailImageBase64(int $idProduct, string $imageFormat = 'jpg'): string
    {

        // 1. Ask PrestaShop for the cover image id
        $idImage = Product::getCover($idProduct)['id_image'] ?? null;
        if (!$idImage) {
            // Fallback: first image
            $images = Image::getImages(Context::getContext()->language->id, $idProduct);
            $idImage = (int)($images[0]['id_image'] ?? 0);
        }

        if (!$idImage) {
            // No image at all – return a transparent 1×1 gif
            return 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
        }

        // 2. Build the physical file path
        $image = new Image($idImage);
        $path  = _PS_PROD_IMG_DIR_ . $image->getExistingImgPath() . '.' . $imageFormat;

        // 3. Pick a size (cart_default ≈ 80×80, change to 'small_default' if you need bigger)
        $type = ImageType::getFormattedName('cart');
        $path = str_replace('.' . $imageFormat, "-{$type}.{$imageFormat}", $path);

        if (!file_exists($path)) {
            // Fallback to the original file
            $path = _PS_PROD_IMG_DIR_ . $image->getExistingImgPath() . '.' . $imageFormat;
        }

        // 4. Read & encode
        $binary = file_get_contents($path);
        if ($binary === false) {
            return 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
        }

        $base64 = base64_encode($binary);
        // return "data:image/{$imageFormat};base64,{$base64}";
        return $base64;
    }



    /**
     * Compares two addresses and returns true if they differ
     * 
     * @param array $address1 First address array with keys: name, contactname, address, address2, zipcode, city, region, country
     * @param array $address2 Second address array with same keys
     * @return bool True if addresses differ, false if identical
     */
    protected function addressesAreDifferent($address1, $address2)
    {
        $fields = ['name', 'address', 'address2', 'zipcode', 'city', 'country'];
        
        foreach ($fields as $field) {
            if (($address1[$field] ?? '') !== ($address2[$field] ?? '')) {
                return true;
            }
        }
        
        return false;
    }




    /**
     * Hook called after an order is validated (new order created).
     * This is where you'll send the order to Picqer.
     */
    public function hookActionValidateOrder($params)
    {
        if(self::DEBUG_LOGGING)
            PrestaShopLogger::addLog(
                'PrestaPicqer: Step 1: Order to be exported to Picqer for Order ID ' . $params['order']->id . '. Step 1: hook triggerd at ' .time(),
                1, null, 'Order', $params['order']->id
            );
        $order_sync_enabled = (bool) Configuration::get(self::PRESTAPICQER_ORDER_SYNC_ENABLED);
        $order_create_statuses = array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_CREATE_STATUSES))));
        $usePsProductPrice = (bool) Configuration::get(self::PRESTAPICQER_USE_PS_PRODUCT_PRICE);
        $createMissingProducts = (bool) Configuration::get(self::PRESTAPICQER_CREATE_MISSING_PRODUCTS);

        $order_process_statuses = array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_PROCESS_STATUSES))));
        $allocateStock = (bool) Configuration::get(self::PRESTAPICQER_ALLOCATE_STOCK);
        $selected_tag_ids = array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_TAGS))));

        if (!$order_sync_enabled || !in_array($params['orderStatus']->id, $order_create_statuses)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Order sync disabled or status not configured for creation. Skipping export for Order ID ' . $params['order']->id. ' Status ID '.$params['orderStatus']->id,
                1, null, 'Order', $params['order']->id
            );
            return;
        }

        $subdomain = Configuration::get(self::PRESTAPICQER_API_SUBDOMAIN);
        $apiKey = Configuration::get(self::PRESTAPICQER_API_KEY);
        $shopIdentifier = Configuration::get(self::PRESTAPICQER_SHOP_IDENTIFIER);

        if (empty($subdomain) || empty($apiKey)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Picqer API Subdomain or Key not configured. Skipping order export for Order ID ' . $params['order']->id,
                2, null, 'Order', $params['order']->id
            );
            return;
        }

        /** @var Order $order */
        $order = $params['order'];
        /** @var Customer $customer */
        $customer = $params['customer'];

        $prestashop_order_note_missing_productnames = [];

        $delivery_address = new Address($order->id_address_delivery);
        $invoice_address = new Address($order->id_address_invoice);

        // Robustly construct delivery name
        $delivery_firstname = (string) $delivery_address->firstname;
        $delivery_lastname = (string) $delivery_address->lastname;
        $delivery_name = trim($delivery_firstname . ' ' . $delivery_lastname);
        if (empty($delivery_name)) {
            // Fallback to customer's name if address name is empty
            $delivery_name = trim((string) $customer->firstname . ' ' . (string) $customer->lastname);
            if (empty($delivery_name)) {
                $delivery_name = 'Unknown Customer'; // Last resort fallback
                PrestaShopLogger::addLog(
                    'PrestaPicqer: Fallback to "Unknown Customer" for delivery_name for Order ID ' . $order->id,
                    2, null, 'Order', $order->id
                );
            }
        }

        // Robustly construct invoice name
        $invoice_firstname = (string) $invoice_address->firstname;
        $invoice_lastname = (string) $invoice_address->lastname;
        $invoice_name = trim($invoice_firstname . ' ' . $invoice_lastname);
        if (empty($invoice_name)) {
            // Fallback to customer's name if address name is empty
            $invoice_name = trim((string) $customer->firstname . ' ' . (string) $customer->lastname);
            if (empty($invoice_name)) {
                $invoice_name = 'Unknown Customer'; // Last resort fallback
                PrestaShopLogger::addLog(
                    'PrestaPicqer: Fallback to "Unknown Customer" for invoice_name for Order ID ' . $order->id,
                    2, null, 'Order', $order->id
                );
            }
        }

        // Try to get pickup_point_data data from sendcloud
        try{
            $sendcloudModule = Module::getInstanceByName('sendcloudv2');
            $cart = new Cart($order->id_cart);
            $servicePoint = $sendcloudModule->hookDisplayBeforeCarrier(['cart' => $cart]);
            $smartyTpl = $sendcloudModule->smarty->tpl_vars;
            if(isset($smartyTpl['service_point_details'])){
                $servicePointData = json_decode($smartyTpl['service_point_details'], true);
                $picqer_pickup_point_data = [
                    // 'pickup_point_data' => [
                        'carrier' => 'SendCloud',
                        'id' => $servicePointData['id'],
                        'name' =>  $servicePointData['name'],
                        'address' => $servicePointData['street'].' '.$servicePointData['house_number'],
                        'zipcode' => $servicePointData['postal_code'],
                        'city' => $servicePointData['city'],
                        'country' => $servicePointData['country'],
                    // ]
                ];
            }
        }
        catch(Exception $e){
            //No sendcloudv2 module or no servicepoint data 
        }

        $picqer_order_reference = Tools::strtoupper($shopIdentifier) . ' #' . $order->id . ' / ' .$order->reference;

        // Check for existing orders in Picqer
        $responseSearchResults = $this->_callPicqerApi('orders?reference=' . urlencode($picqer_order_reference), 'GET');
        if(isset($responseSearchResults) && is_array($responseSearchResults)){
            foreach($responseSearchResults as $responseSearchResult){
                if(isset($responseSearchResult['reference']) && $responseSearchResult['reference'] == $picqer_order_reference){
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Failed to create order '.$order->id.' in Picqer. Found existing order with identical order reference ('.$picqer_order_reference.').',
                        3, null, 'Order', $order->id
                    );
                    return;
                }
            }
        }

        //Get language code for this order
        $languageCode = 'en';
        $id_lang = (int)$order->id_lang;
        if ($id_lang){
            $language = new Language($id_lang);
            if (Validate::isLoadedObject($language) && !empty($language->iso_code) && in_array(strtolower($language->iso_code), ['en', 'nl'])){
                    $languageCode = strtolower($language->iso_code);
            }
        }

        $picqer_order_data = [
            'reference' => $picqer_order_reference,
            'deliveryname' => $delivery_name,
            'deliveryaddress' => $delivery_address->address1,
            'deliveryaddress2' => $delivery_address->address2,
            'deliveryzipcode' => $delivery_address->postcode,
            'deliverycity' => $delivery_address->city,
            'deliverycountry' => Country::getIsoById($delivery_address->id_country),
            'invoicename' => $invoice_name,
            'invoiceaddress' => $invoice_address->address1,
            'invoiceaddress2' => $invoice_address->address2,
            'invoicezipcode' => $invoice_address->postcode,
            'invoicecity' => $invoice_address->city,
            'invoicecountry' => Country::getIsoById($invoice_address->id_country),
            'emailaddress' => $customer->email,
            'telephone' => $delivery_address->phone ?? $invoice_address->phone,
            'language' => $languageCode,
           
            'products' => [],
            
            //Can only be used with a customer
            // 'vatnumber' => $customer->vat_number
            // 'contactname' => if used with a company name in the 'name' field
        ];


        // Sync customer with Picqer if enabled
        $createCustomersEnabled = (bool) Configuration::get(self::PRESTAPICQER_CREATE_CUSTOMERS_ENABLED);
        if ($createCustomersEnabled) {
            $customer_email = (string) $customer->email;
            
            if (!empty($customer_email)) {
                // Search for existing customer by email
                $picqer_customers = $this->_callPicqerApi('customers?search=' . urlencode($customer_email), 'GET');
                $picqer_customer_id = null;
                $picqer_customer = null;
                
                if (is_array($picqer_customers)) {
                    foreach ($picqer_customers as $pc) {
                        if (isset($pc['emailaddress']) && $pc['emailaddress'] === $customer_email) {
                            $picqer_customer = $pc;
                            $picqer_customer_id = (int) $pc['idcustomer'];
                            break;
                        }
                    }
                }
                
                if ($picqer_customer_id) {
                    // Customer exists - update if needed
                    $update_data = [];
                    
                    // Compare customer fields
                    if (($picqer_customer['name'] ?? '') !== $delivery_name) {
                        $update_data['name'] = $delivery_name;
                    }
                    if (($picqer_customer['telephone'] ?? '') !== ($delivery_address->phone ?? $invoice_address->phone ?? '')) {
                        $update_data['telephone'] = $delivery_address->phone ?? $invoice_address->phone ?? '';
                    }
                    if (($picqer_customer['vatnumber'] ?? '') !== ($customer->vat_number ?? '')) {
                        $update_data['vatnumber'] = $customer->vat_number ?? '';
                    }
                    if (($picqer_customer['language'] ?? '') !== $languageCode) {
                        $update_data['language'] = $languageCode;
                    }
                    
                    if (!empty($update_data)) {
                        $this->_callPicqerApi('customers/' . $picqer_customer_id, 'PUT', $update_data);
                    }
                    
                    // Prepare address data for comparison
                    $ps_delivery_address_data = [
                        'name' => $picqer_order_data['deliveryname'],
                        'address' => $picqer_order_data['deliveryaddress'],
                        'address2' => $picqer_order_data['deliveryaddress2'],
                        'zipcode' => $picqer_order_data['deliveryzipcode'],
                        'city' => $picqer_order_data['deliverycity'],
                        'country' => $picqer_order_data['deliverycountry'],
                    ];
                    
                    $ps_invoice_address_data = [
                        'name' => $picqer_order_data['invoicename'],
                        'address' => $picqer_order_data['invoiceaddress'],
                        'address2' => $picqer_order_data['invoiceaddress2'],
                        'zipcode' => $picqer_order_data['invoicezipcode'],
                        'city' => $picqer_order_data['invoicecity'],
                        'country' => $picqer_order_data['invoicecountry'],
                    ];
                    
                    // Find and compare default delivery address
                    $picqer_addresses = isset($picqer_customer['addresses']) ? $picqer_customer['addresses'] : [];
                    $default_delivery_address = null;
                    $default_invoice_address = null;
                    
                    foreach ($picqer_addresses as $addr) {
                        if ($addr['defaultdelivery'] ?? false) {
                            $default_delivery_address = $addr;
                        }
                        if ($addr['defaultinvoice'] ?? false) {
                            $default_invoice_address = $addr;
                        }
                    }
                    
                    // Update or create delivery address
                    if ($default_delivery_address && $this->addressesAreDifferent($default_delivery_address, $ps_delivery_address_data)) {
                        $this->_callPicqerApi(
                            'customers/' . $picqer_customer_id . '/addresses/' . $default_delivery_address['idcustomer_address'],
                            'PUT',
                            $ps_delivery_address_data
                        );
                    } elseif (!$default_delivery_address) {
                        $ps_delivery_address_data['defaultdelivery'] = true;
                        $this->_callPicqerApi('customers/' . $picqer_customer_id . '/addresses', 'POST', $ps_delivery_address_data);
                    }
                    
                    // Update or create invoice address (if different from delivery)
                    if ($invoice_address->id !== $delivery_address->id) {
                        if ($default_invoice_address && $this->addressesAreDifferent($default_invoice_address, $ps_invoice_address_data)) {
                            $this->_callPicqerApi(
                                'customers/' . $picqer_customer_id . '/addresses/' . $default_invoice_address['idcustomer_address'],
                                'PUT',
                                $ps_invoice_address_data
                            );
                        } elseif (!$default_invoice_address) {
                            $ps_invoice_address_data['defaultinvoice'] = true;
                            $this->_callPicqerApi('customers/' . $picqer_customer_id . '/addresses', 'POST', $ps_invoice_address_data);
                        }
                    }
                    
                    $picqer_order_data['idcustomer'] = $picqer_customer_id;
                } else {
                    // Customer doesn't exist - create new one
                    $new_customer_data = [
                        'email' => $customer_email,
                        'name' => $delivery_name,
                        'telephone' => $delivery_address->phone ?? $invoice_address->phone ?? '',
                        'vatnumber' => $customer->vat_number ?? '',
                        'language' => $languageCode,
                    ];
                    
                    $create_response = $this->_callPicqerApi('customers', 'POST', $new_customer_data);
                    
                    if ($create_response && isset($create_response['idcustomer'])) {
                        $new_picqer_customer_id = (int) $create_response['idcustomer'];
                        
                        // Add delivery address
                        $ps_delivery_address_data = [
                            'name' => $delivery_name,
                            'contactname' => '',
                            'address' => $delivery_address->address1,
                            'address2' => $delivery_address->address2 ?? '',
                            'zipcode' => $delivery_address->postcode,
                            'city' => $delivery_address->city,
                            'region' => '',
                            'country' => Country::getIsoById($delivery_address->id_country),
                            'defaultdelivery' => true,
                        ];
                        $this->_callPicqerApi('customers/' . $new_picqer_customer_id . '/addresses', 'POST', $ps_delivery_address_data);
                        
                        // Add invoice address if different
                        if ($invoice_address->id !== $delivery_address->id) {
                            $ps_invoice_address_data = [
                                'name' => $invoice_name,
                                'contactname' => '',
                                'address' => $invoice_address->address1,
                                'address2' => $invoice_address->address2 ?? '',
                                'zipcode' => $invoice_address->postcode,
                                'city' => $invoice_address->city,
                                'region' => '',
                                'country' => Country::getIsoById($invoice_address->id_country),
                                'defaultinvoice' => true,
                            ];
                            $this->_callPicqerApi('customers/' . $new_picqer_customer_id . '/addresses', 'POST', $ps_invoice_address_data);
                        }
                        
                        $picqer_order_data['idcustomer'] = $new_picqer_customer_id;
                    }
                }
            }
        }



        if(isset($servicePointData)){
            $picqer_order_data['pickup_point_data'] = $picqer_pickup_point_data;
        }


        // Add customer remark 
        // $sql = 'SELECT message FROM `' . _DB_PREFIX_ . 'message` WHERE id_order='.$order->id.' AND private = 0 ORDER BY id_message ASC LIMIT 1;';
        // $order_messages = Db::getInstance()->executeS($sql);
        // if(isset($order_messages[0]['message']) && !empty($order_messages[0]['message'])){
        //     $picqer_order_data['customer_remarks'] = $order_messages[0]['message'];
        // }

        $message = new Message();
        $order_messages = $message->getMessagesByOrderId((int)$order->id);
        foreach($order_messages as $order_message){
            if($order_message['private'] == 0 && !empty($order_message['message'])){
                $picqer_order_data['customer_remarks'] = $order_message['message'];
                break;
            }
        }

        // Shipping provider
        // $carrier = new Carrier((int)$order->id_carrier);
        // $picqer_shippingproviders = $this->_callPicqerApi('shippingproviders');
        // if(is_array($picqer_shippingproviders)){
        //     foreach($picqer_shippingproviders as $picqer_shippingprovider){
        //         foreach($picqer_shippingprovider['profiles'] as $profile){
        //             if($profile['name'] == $carrier->name){
        //                 $picqer_order_data['idshippingprovider_profile'] = $profile['idshippingprovider_profile'];
        //                 break 2;
        //             }
        //         }
        //     }
        // }
        // if(!isset($picqer_order_data['idshippingprovider_profile']) && !empty($carrier->name)){
        //     PrestaShopLogger::addLog(
        //         'PrestaPicqer: Could not find shipping provider for carrier '.$carrier->name.' For order ID: ' . $order->id . '. Order will be sent to Picqer without shipping provider.',
        //         3, null, 'Order', $order->id
        //     );
        // }




        if ($usePsProductPrice) {
            $picqerVatGroups = $this->_callPicqerApi('vatgroups');
        }

        $products = $order->getProducts();
        foreach ($products as $product) {

            $picqer_product = [
                'productcode' => !empty($product['product_reference']) ? $product['product_reference'] : '',
                'name' => $product['product_name'],
                'amount' => (int) $product['product_quantity'],
            ];

            if ($usePsProductPrice) {
                $picqer_product['price'] = (float) $product['unit_price_tax_excl'];

                $prestashopVatPercentage = (float) $product['tax_rate'];
                if(is_array($picqerVatGroups)){
                    foreach($picqerVatGroups as $vatGroup){
                        $vatGroupPercentage = (float) $vatGroup['percentage'];
                        if($vatGroupPercentage === $prestashopVatPercentage){
                            $picqer_product['idvatgroup'] = $vatGroup['idvatgroup'];
                            break;
                        }
                    }
                }

                if(!isset($picqer_product['idvatgroup'])){
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Could not find VAT group "'.$product['tax_rate'].'" for product ' . $picqer_product['productcode'] . '. Order ID: ' . $order->id . '. Order will be sent to Picqer without VAT group for this product.',
                        2, null, 'Product', $product['product_id']
                    );
                }
            }
            

            // Get the Picqer idproduct from search for the product code
            $product_code_to_search = $picqer_product['productcode'];
           
            $found_picqer_idproduct = 0;
            // Lookup Picqer's idproduct
            if(!empty($product_code_to_search)){
                $picqer_product_lookup = $this->_callPicqerApi('products?search=' . urlencode($product_code_to_search), 'GET');

                if(self::DEBUG_LOGGING)
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Step 2: Finished searching for product in Picqer ' . $product['product_reference'] . ' at ' .time(),
                        1, null, 'Order', $order->id
                    );

                if(is_array($picqer_product_lookup)) {
                    foreach ($picqer_product_lookup as $picqer_item) {
                        if (isset($picqer_item['productcode']) && $picqer_item['productcode'] === $product_code_to_search) {
                            $found_picqer_idproduct = (int) $picqer_item['idproduct'];
                            break;
                        }
                    }
                }
            }

            if ($found_picqer_idproduct > 0) {
                $picqer_product['idproduct'] = $found_picqer_idproduct;
                $picqer_order_data['products'][] = $picqer_product;
            } else {
                if($createMissingProducts && !empty($picqer_product['productcode'])){

                    $newPicqerProduct = [
                        'productcode' => $picqer_product['productcode'],
                        'name' => $picqer_product['name'],
                        'idvatgroup' => $picqer_product['idvatgroup'],
                        'price' => (float) $product['original_product_price'],
                        // 'price' => (float) $product['unit_price_tax_excl'],
                    ];

                    //Double check the VAT group
                    if(!isset($newPicqerProduct['idvatgroup']) || empty($newPicqerProduct['idvatgroup'])){
                        if(!isset($picqerVatGroups)){
                            $picqerVatGroups = $this->_callPicqerApi('vatgroups');
                        }
                        $prestashopVatPercentage = (float) $product['tax_rate'];
                        if(is_array($picqerVatGroups)){
                            foreach($picqerVatGroups as $vatGroup){
                                $vatGroupPercentage = (float) $vatGroup['percentage'];
                                if($vatGroupPercentage === $prestashopVatPercentage){
                                // if((float) $vatGroup['percentage'] === (float) $product['tax_rate']){
                                    $newPicqerProduct['idvatgroup'] = $vatGroup['idvatgroup'];
                                    break;
                                }
                            }
                        }

                        if(!isset($newPicqerProduct['idvatgroup'])){
                            PrestaShopLogger::addLog(
                                'PrestaPicqer: Could not find VAT group "'.$product['tax_rate'].'" for product ' . $picqer_product['productcode'] . '. While creating new product in Picqer. Order ID: ' . $order->id . '. Product will be created in Picqer without VAT group.',
                                2, null, 'Product', $product['product_id']
                            );
                        }
                    }

                    $newProductResponse = $this->_callPicqerApi('products', 'POST', $newPicqerProduct);
                    if(isset($newProductResponse['idproduct'])){
                        $picqer_product['idproduct'] = $newProductResponse['idproduct'];
                        $picqer_order_data['products'][] = $picqer_product;
                        
                        PrestaShopLogger::addLog(
                            'PrestaPicqer: Created product for "'.$picqer_product['name'].'" with productcode ' . $product_code_to_search . ' in Picqer.',
                            1, null, 'Product', $product['product_id']
                        ); 

                        // Add product image
                        $productImage = $this->getOrderDetailImageBase64($product['product_id']);
                        $productImageResponse = $this->_callPicqerApi('products/'.$picqer_product['idproduct'].'/images', 'POST', ['image' => $productImage]);
                        
                        if(isset($productImageResponse['idproduct_image'])){
                            PrestaShopLogger::addLog(
                            'PrestaPicqer: Created product image for "'.$picqer_product['name'].'" with productcode ' . $product_code_to_search . ' in Picqer.',
                            1, null, 'Product', $product['product_id']
                            ); 
                        }
                        else{
                             PrestaShopLogger::addLog(
                            'PrestaPicqer: Could not create product image for "'.$picqer_product['name'].'" with productcode ' . $product_code_to_search . ' in Picqer.',
                            2, null, 'Product', $product['product_id']
                            ); 
                        }

                        //Add stock to new product
                        if(!isset($picqer_warehouses[0]['idwarehouse'])){
                            $picqer_warehouses = $this->_callPicqerApi('warehouses', 'GET');
                        }
                        if(isset($picqer_warehouses[0]['idwarehouse'])){
                            $pproduct = new Product($product['product_id']);
                            $pproduct->loadStockData(); // fills $product->quantity
                            $picqer_add_stock_response = $this->_callPicqerApi('products/'.$picqer_product['idproduct'].'/stock/'.$picqer_warehouses[0]['idwarehouse'], 'POST', [
                                'idlocation' => null,
                                'change' => $pproduct->quantity,
                                'reason' => 'Initial Prestapicqer stock level'
                            ]);
                            if(!isset($picqer_add_stock_response['stock'])){
                                PrestaShopLogger::addLog(
                                'PrestaPicqer: Could not add stock for "'.$picqer_product['name'].'" with productcode ' . $product_code_to_search . ' in Picqer.',
                                2, null, 'Product', $product['product_id']
                                ); 
                            }
                        }

                        usleep(500000); //Small .5 second delay for Picqer to process the new product.

                    }
                    else{
                        PrestaShopLogger::addLog(
                            'PrestaPicqer: Could not create product "'.$picqer_product['name'].'" with productcode ' . $product_code_to_search . ' in Picqer. Order ' . $order->id . ' will be sent without this item.',
                        3, null, 'Product', $product['product_id']
                        ); 
                    }
                }
                else{
                    $prestashop_order_note_missing_productnames[] = $picqer_product['name'];
                    if(empty($product_code_to_search)){
                        PrestaShopLogger::addLog(
                            'PrestaPicqer: Could not find product "'.$picqer_product['name'].'" product inPicqer, product has no productcode. Order ' . $order->id . ' will be sent without this item.',
                            2, null, 'Product', $product['product_id']
                        );
                    }
                    else{
                        PrestaShopLogger::addLog(
                            'PrestaPicqer: Could not find product "'.$picqer_product['name'].'" with productcode ' . $product_code_to_search . ' in Picqer. Order ' . $order->id . ' will be sent without this item.',
                            2, null, 'Product', $product['product_id']
                        );
                    }
                }
            }
        }

        // Check if products array is empty before sending
        if (empty($picqer_order_data['products'])) {
            $no_products_note = 'No matching products found in Picqer for Order ID ' . $order->id . '. Skipped export to Picqer.';
            PrestaShopLogger::addLog(
                'PrestaPicqer: '.$no_products_note,
                2, null, 'Order', $order->id
            );
            $this->addOrderMessage($order, "⚠️ WARNING ⚠️\n".$no_products_note);
            // $order->note = "⚠️ WARNING ⚠️\n".$no_products_note;

            
            $order->save();
            return;
        }

        $response = $this->_callPicqerApi('orders', 'POST', $picqer_order_data);

        if(self::DEBUG_LOGGING)
            PrestaShopLogger::addLog(
                'PrestaPicqer: Step 3: Order created in Picqer ' . $order->id . ' at ' . time(),
                1, null, 'Order', $order->id
            );
        if ($response && isset($response['idorder'])) {

            // Add selected Picqer tags to the order
            if (!empty($selected_tag_ids)) {
                $tagResponsesErrors = [];
                foreach($selected_tag_ids as $selected_tag_id){
                    $tagsResponse = $this->_callPicqerApi('orders/'.$response['idorder'].'/tags', 'POST', ['idtag' => $selected_tag_id]);
                    if(!(isset($tagsResponse['idtag']) && $tagsResponse['idtag'] == $selected_tag_id))
                        $tagResponsesErrors[] = $tagsResponse;
                }
                if(!empty($tagResponsesErrors)){
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Failed to add tags to Picqer for Order ID ' . $order->id . '.',
                        2, null, 'Order', $order->id
                    );
                }
            }

            //Set Picqer orderfields
            $picqer_orderfields = $this->_callPicqerApi('orderfields');
            if(is_array($picqer_orderfields)){
                foreach($picqer_orderfields as $pof){
                    switch($pof['title']){
                        case 'shipping method':
                            $carrier = new Carrier((int)$order->id_carrier);
                            $picker_orderfield_resonse = $this->_callPicqerApi('orders/'.$response['idorder'].'/orderfields/'.$pof['idorderfield'], 'PUT', ['value' => $carrier->name]);
                            if(!isset($picker_orderfield_resonse['idorderfield'])){
                                PrestaShopLogger::addLog(
                                    'PrestaPicqer: Failed to set orderfield '.$pof['title']. '. For order ' . $response['orderid'] . ' in Picqer. Response: ' . json_encode($picker_orderfield_resonse),
                                    3, null, 'Order', $order->id
                                );
                            }
                        break;
                        case 'shipping cost':
                            $picker_orderfield_resonse = $this->_callPicqerApi('orders/'.$response['idorder'].'/orderfields/'.$pof['idorderfield'], 'PUT', ['value' => $order->total_shipping_tax_excl]);
                            if(!isset($picker_orderfield_resonse['idorderfield'])){
                                PrestaShopLogger::addLog(
                                    'PrestaPicqer: Failed to set orderfield '.$pof['title']. '. For order ' . $response['orderid'] . ' in Picqer. Response: ' . json_encode($picker_orderfield_resonse),
                                    3, null, 'Order', $order->id
                                );
                            }
                        break;
                        case 'payment method':
                            $picker_orderfield_resonse = $this->_callPicqerApi('orders/'.$response['idorder'].'/orderfields/'.$pof['idorderfield'], 'PUT', ['value' => $order->payment]);
                            if(!isset($picker_orderfield_resonse['idorderfield'])){
                                PrestaShopLogger::addLog(
                                    'PrestaPicqer: Failed to set orderfield '.$pof['title']. '. For order ' . $response['orderid'] . ' in Picqer. Response: ' . json_encode($picker_orderfield_resonse),
                                    3, null, 'Order', $order->id
                                );
                            }
                        break;
                    }
                }
            }

            if(in_array($params['orderStatus']->id, $order_process_statuses)){
                $process_order_response = $this->_callPicqerApi('orders/'.$response['idorder'].'/process', 'POST');
                if(!(isset($process_order_response['idorder']) && $process_order_response['idorder'] == $response['idorder'])){
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Failed to process order ' . $response['orderid'] . ' in Picqer. Response: ' . json_encode($process_order_response),
                        3, null, 'Order', $order->id
                    );
                }
            }
            elseif($allocateStock){ //Allocate stock for new concept order
                $allocate_stock_response = $this->_callPicqerApi('orders/'.$response['idorder'].'/allocate', 'POST');
                if(!(isset($allocate_stock_response['idorder']) && $allocate_stock_response['idorder'] == $response['idorder'])){
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Failed to allocate stock for order ' . $response['orderid'] . ' in Picqer. Response: ' . json_encode($allocate_stock_response),
                        3, null, 'Order', $order->id
                    );
                }
            }


            // Add Picqer's internal order ID to PrestaShop order note
            $prestashop_order_note = 'Picqer order '.$response['orderid'].' created.';
            if(!empty($prestashop_order_note_missing_productnames)){
                $missing_products_warning =  "\n⚠️ WARNING ⚠️\nThese products were ordered but not found in Picqer!\n- ".implode("\n- ", $prestashop_order_note_missing_productnames);
                $prestashop_order_note .= $missing_products_warning;

                $comments_response = $this->_callPicqerApi('orders/'.$response['idorder'].'/comments', 'POST', [
                    'body' => $missing_products_warning,
                    'show_at_related' => true  //shows on picklist now
                ]);

            }
            $this->addOrderMessage($order, $prestashop_order_note);
            // $order->note = $prestashop_order_note;
            
            $order->save();

           

            PrestaShopLogger::addLog(
                'PrestaPicqer: Successfully exported Order ID ' . $order->id . ' to Picqer. Picqer ID: ' . $response['orderid'],
                1, null, 'Order', $order->id
            );
        } else {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Failed to export Order ID ' . $order->id . ' to Picqer. Request: ' . json_encode($picqer_order_data) . '. Response: ' . json_encode($response),
                3, null, 'Order', $order->id
            );
        }

        if(self::DEBUG_LOGGING)
            PrestaShopLogger::addLog(
                'PrestaPicqer: Step 4: Order processing finished ' . $order->id . ' at ' . time(),
                1, null, 'Order', $order->id
            );
    }

    /**
     * Hook called when an order's status is updated.
     * Update the order status in Picqer to 'processing' or 'cancelled'.
     */
    public function hookActionOrderStatusUpdate($params)
    {
        $order_sync_enabled = (bool) Configuration::get(self::PRESTAPICQER_ORDER_SYNC_ENABLED);
        $order_process_statuses = array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_PROCESS_STATUSES))));
        $order_cancel_statuses = array_map('intval', array_filter(explode(',', Configuration::get(self::PRESTAPICQER_ORDER_CANCEL_STATUSES))));

        if (!$order_sync_enabled) {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Order sync disabled. Skipping Picqer status update for Order ID ' . $params['id_order'],
                1, null, 'Order', $params['id_order']
            );
            return;
        }

        $subdomain = Configuration::get(self::PRESTAPICQER_API_SUBDOMAIN);
        $apiKey = Configuration::get(self::PRESTAPICQER_API_KEY);
        $shopIdentifier = Configuration::get(self::PRESTAPICQER_SHOP_IDENTIFIER);

        if (empty($subdomain) || empty($apiKey)) {
            PrestaShopLogger::addLog(
                'PrestaPicqer: Picqer API Subdomain or Key not configured. Skipping order status update for Order ID ' . $params['id_order'],
                2, null, 'Order', $params['id_order']
            );
            return;
        }

        /** @var Order $order */
        $order = new Order((int) $params['id_order']);
        $picqer_order_reference = Tools::strtoupper($shopIdentifier) . ' #' . $order->id . ' / ' .$order->reference;

        $target_picqer_status = null;

        // Prioritize 'cancelled' status if the new PrestaShop status is configured for it
        if (in_array($params['newOrderStatus']->id, $order_cancel_statuses)) {
            $target_picqer_status = 'Cancelled';
        } elseif (in_array($params['newOrderStatus']->id, $order_process_statuses)) {
            $target_picqer_status = 'Process';
        } 

        // If a target status is determined, make the API call
        if ($target_picqer_status !== null) {


            $retries = 2;
            do{
                $picqer_id_order = null;
                $picqer_current_status = null;
                $responseSearchResults = $this->_callPicqerApi('orders?reference=' . urlencode($picqer_order_reference), 'GET');
                if(isset($responseSearchResults) && is_array($responseSearchResults)){
                    foreach($responseSearchResults as $responseSearchResult){
                        if(isset($responseSearchResult['reference']) && $responseSearchResult['reference'] == $picqer_order_reference){
                            $picqer_id_order = $responseSearchResult['idorder'];
                            $picqer_current_status = $responseSearchResult['status'];
                            break 2;
                        }
                    }
                }
                $retries --;
                sleep(2);
            } while($retries > 0 && empty($picqer_id_order));

            if(empty($picqer_id_order)){
                PrestaShopLogger::addLog(
                    'PrestaPicqer: Failed to update order status. Failed to find order '.$order->id.' in Picqer ('.$picqer_order_reference.').',
                    3, null, 'Order', $order->id
                );
                return;
            }

            if($target_picqer_status == 'Process' && $picqer_current_status != 'processing'){
                $response = $this->_callPicqerApi('orders/' . $picqer_id_order . '/process', 'POST', []);
                if (isset($response['idorder'])) {
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Picqer Order Number: ' . $picqer_order_reference . '. Successfully updated status to processing.',
                        1, null, 'Order', $order->id
                    );
                } else {
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Picqer Order Number: ' . $picqer_order_reference . ' failed to update status to processing. Response: ' . json_encode($response),
                        3, null, 'Order', $order->id
                    );
                }
            }
            elseif($target_picqer_status == 'Cancelled'){
                $response = $this->_callPicqerApi('orders/' . $picqer_id_order .'?force=true', 'DELETE');
                if (empty($response)) {
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Picqer Order Number: ' . $picqer_order_reference . '. Successfully updated status to cancelled.',
                        1, null, 'Order', $order->id
                    );
                } else {
                    PrestaShopLogger::addLog(
                        'PrestaPicqer: Picqer Order Number: ' . $picqer_order_reference . ' failed to update status to cancelled. Response: ' . json_encode($response),
                        3, null, 'Order', $order->id
                    );
                }
            }
        }
    }
}